<?php

/**
 * Inventory Currency Field Class.
 *
 * @package   InventoryField
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\Exceptions\AppException;
use App\Exceptions\Security;
use App\Fields\Currency;
use App\Inventory\Currency\Param;
use App\Json;
use App\Purifier;
use App\TextUtils;
use vtlib\Functions;
use yii\db\Schema;

class Vtiger_Currency_InventoryField extends Vtiger_Basic_InventoryField
{
	protected $type = 'Currency';
	protected $defaultLabel = 'LBL_CURRENCY';
	protected $columnName = 'currency';
	protected $dbType = [Schema::TYPE_INTEGER, 11];
	protected $customColumn = [
		'currencyparam' => [Schema::TYPE_STRING, 1024],
	];
	protected $blocks = [0];
	protected $maximumLength = '-2147483648,2147483647';
	protected $customMaximumLength = [
		'currencyparam' => 1024
	];
	protected $purifyType = Purifier::INTEGER;
	protected $customPurifyType = [
		'currencyparam' => Purifier::TEXT
	];
	/** {@inheritdoc} */
	protected $params = ['reset_currency', 'cit_exchange'];

	/** {@inheritdoc} */
	public function getDisplayValue($value, array $rowData = [], bool $rawText = false)
	{
		if (empty($value)) {
			return '';
		}
		return Currency::getById($value)['currency_name'];
	}

	/**
	 * Gets currency param.
	 *
	 * @param array       $currencies
	 * @param string|null $param
	 * @param string      $date
	 *
	 * @throws AppException
	 *
	 * @return array
	 */
	public function getCurrencyParam(array $currencies, ?string $param = '', string $date = '')
	{
		$params = [];
		if ($param) {
			$params = Json::decode($param);
		}
		foreach ($currencies as $currency) {
			if (!isset($params[$currency['id']])) {
				$params[$currency['id']] = Functions::getConversionRateInfo($currency['id'], $date);
			}
		}
		return $params;
	}

	/**
	 * Returns currency param object representation.
	 *
	 * @param string $jsonParam
	 *
	 * @throws AppException
	 *
	 * @return Param
	 */
	public function createCurrencyParamObject(string $jsonParam): Param
	{
		return Param::fromJson($jsonParam);
	}

	/** {@inheritdoc} */
	public function getDBValue($value, ?string $name = '')
	{
		if ($name === $this->getColumnName()) {
			$value = (int) $value;
		}
		return $value;
	}

	/** {@inheritdoc} */
	public function getDefaultValue(string $columnName = '')
	{
		$value = parent::getDefaultValue($columnName);
		if (!$value) {
			if (!$columnName || $columnName === $this->getColumnName()) {
				$value = Currency::getDefault()['id'];
			} else {
				$value = Json::encode($this->getCurrencyParam(Currency::getAll(true)));
			}
		}

		return $value;
	}

	/** {@inheritdoc} */
	public function validate($value, string $columnName, bool $isUserFormat, $originalValue = null)
	{
		if ($columnName === $this->getColumnName()) {
			if (!is_numeric($value) || !isset(Currency::getAll()[$value])) {
				throw new Security("ERR_ILLEGAL_FIELD_VALUE||$columnName||" . print_r($value, true), 406);
			}
		} elseif (!\is_array($value) && TextUtils::getTextLength($value) > $this->customMaximumLength[$columnName]) {
			throw new Security("ERR_VALUE_IS_TOO_LONG||$columnName||" . print_r($value, true), 406);
		}
	}

	/** {@inheritdoc} */
	public function compare($value, $prevValue, string $column): bool
	{
		return $column === $this->getColumnName() ? (int) $value === (int) $prevValue : parent::compare($value, $prevValue, $column);
	}

	/** {@inheritdoc} */
	public function getConfigFieldsData(): array
	{
		$data = parent::getConfigFieldsData();
		$data['reset_currency'] = [
			'name' => 'reset_currency',
			'label' => 'LBL_INV_CURRENCY_RESET',
			'uitype' => 56,
			'maximumlength' => '1',
			'typeofdata' => 'C~O',
			'purifyType' => Purifier::INTEGER,
			'defaultvalue' => 0,
			'tooltip' => 'LBL_INV_CURRENCY_RESET_DESC'
		];
		$data['cit_exchange'] = [
			'name' => 'cit_exchange',
			'label' => 'LBL_INV_ADDITIONAL_CIT_EXCHANGE',
			'uitype' => 56,
			'maximumlength' => '1',
			'typeofdata' => 'C~O',
			'purifyType' => Purifier::INTEGER,
			'defaultvalue' => 0,
			'tooltip' => 'LBL_INV_ADDITIONAL_CIT_EXCHANGE_DESC'
		];

		return $data;
	}

	/**
	 * Returns currency conversation summary grouped by tax groups.
	 *
	 * @param Vtiger_Record_Model $model
	 *
	 * @return array
	 */
	public function getCurrencyConversationSummary(Vtiger_Record_Model $model): array
	{
		$inventoryModel = Vtiger_Inventory_Model::getInstance($this->getModuleName());
		$currency = $inventoryModel->getCurrency($model);
		$defaultCurrency = Currency::getDefault();

		$taxSummary = static::getInstance($this->getModuleName(), 'Tax')->getTaxSummary($model);
		if (
			$currency['id'] === $defaultCurrency['id']
			|| !$taxSummary
		) {
			return [];
		}

		$inventory = $model->getInventoryData();
		$row = key($inventory);

		$currencyParam = $this->getCurrencyParam([], $inventory[$row]['currencyparam']);
		$conversionRate = (float) $currencyParam[$defaultCurrency['id']]['conversion'] / (float) $currencyParam[(int) $inventory[$row]['currency']]['conversion'];

		$groups = [];
		$total = 0;
		foreach ($taxSummary['groups'] as $taxGroup => $value) {
			$result = $this->roundDecimal($value * $conversionRate);
			$groups[$taxGroup] = $result;
			$total += $result;
		}

		return [
			'groups' => $groups,
			'total' => $this->roundDecimal($total),
		];
	}
}
