<?php

/**
 * UIType virtual field file.
 *
 * @package   UIType
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

use App\Language;
use App\Log;
use App\Record;

/**
 * UIType virtual field class.
 */
class Vtiger_Virtual_UIType extends Vtiger_Base_UIType
{
	/** @var Vtiger_Field_Model|null Virtual field model */
	protected $virtualFieldModel;
	/** @var Vtiger_Record_Model[] Virtual record model */
	protected static $virtualRecordModel;
	/** @var string Base field name */
	protected $baseField;
	/** @var string Reference module name */
	protected $refModuleName;
	/** @var string Field name from reference module */
	protected $refFieldName;

	/** {@inheritdoc} */
	public function __construct($fieldModel)
	{
		parent::__construct($fieldModel);
		$fieldParams = $this->getFieldModel()->getFieldParams();
		[$this->baseField,$this->refModuleName,$this->refFieldName] = explode('::', $fieldParams['virtualField']);
		$this->virtualFieldModel = Vtiger_Module_Model::getInstance($this->refModuleName)->getField($this->refFieldName);
		if (empty($this->virtualFieldModel)) {
			Log::error("The virtual field does not exist Field name: {$this->getFieldModel()->getName()}, virtual field: {$fieldParams['virtualField']}", 'UIType');
		}
	}

	/**
	 * Get virtual field model.
	 *
	 * @return Vtiger_Field_Model
	 */
	public function getVirtualFieldModel(): ?Vtiger_Field_Model
	{
		return $this->virtualFieldModel;
	}

	/**
	 * Get virtual base field name.
	 *
	 * @return string
	 */
	public function getVirtualBaseField(): string
	{
		return $this->baseField;
	}

	/** {@inheritdoc} */
	public function getRangeValues()
	{
		return '';
	}

	/** {@inheritdoc} */
	public function isAjaxEditable()
	{
		return false;
	}

	/** {@inheritdoc} */
	public function getDisplayValue($value, $record = false, $recordModel = false, $rawText = false, $length = false)
	{
		$refRecordModel = $this->getRecordModel($recordModel);
		if (empty($this->virtualFieldModel)) {
			return $value;
		}

		return $this->virtualFieldModel->getUITypeModel()->getDisplayValue(
			$refRecordModel ? $refRecordModel->get($this->refFieldName) : $value,
			$refRecordModel ? $refRecordModel->getId() : false,
			$refRecordModel ?: false,
			$rawText,
			$length
		);
	}

	/** {@inheritdoc} */
	public function getListViewDisplayValue($value, $record = false, $recordModel = false, $rawText = false)
	{
		$refRecordModel = $this->getRecordModel($recordModel);
		if (null === $refRecordModel || empty($this->virtualFieldModel)) {
			return $value;
		}
		return $this->virtualFieldModel->getUITypeModel()->getListViewDisplayValue(
			$refRecordModel->get($this->refFieldName),
			$refRecordModel->getId(),
			$refRecordModel,
			$rawText,
		);
	}

	/** {@inheritdoc} */
	public function getTilesDisplayValue($value, $record = false, $recordModel = false, $rawText = false)
	{
		$refRecordModel = $this->getRecordModel($recordModel);
		if (null === $refRecordModel || empty($this->virtualFieldModel)) {
			return $value;
		}
		return $this->virtualFieldModel->getUITypeModel()->getTilesDisplayValue(
			$refRecordModel->get($this->refFieldName),
			$refRecordModel->getId(),
			$refRecordModel,
			$rawText,
		);
	}

	/** {@inheritdoc} */
	public function getRelatedListViewDisplayValue($value, $record = false, $recordModel = false, $rawText = false)
	{
		$refRecordModel = $this->getRecordModel($recordModel);
		if (null === $refRecordModel || empty($this->virtualFieldModel)) {
			return $value;
		}
		return $this->virtualFieldModel->getUITypeModel()->getRelatedListViewDisplayValue(
			$refRecordModel->get($this->refFieldName),
			$refRecordModel->getId(),
			$refRecordModel,
			$rawText,
		);
	}

	/** {@inheritdoc} */
	public function getHistoryDisplayValue($value, Vtiger_Record_Model $recordModel, $rawText = false)
	{
		$refRecordModel = $this->getRecordModel($recordModel);
		if (null === $refRecordModel || empty($this->virtualFieldModel)) {
			return $value;
		}
		return $this->virtualFieldModel->getUITypeModel()->getHistoryDisplayValue(
			$refRecordModel->get($this->refFieldName),
			$refRecordModel,
			$rawText,
		);
	}

	/** {@inheritdoc} */
	public function getTextParserDisplayValue($value, Vtiger_Record_Model $recordModel, $params)
	{
		$refRecordModel = $this->getRecordModel($recordModel);
		if (null === $refRecordModel || empty($this->virtualFieldModel)) {
			return $value;
		}
		return $this->virtualFieldModel->getUITypeModel()->getTextParserDisplayValue(
			$refRecordModel->get($this->refFieldName),
			$refRecordModel,
			$params,
		);
	}

	/** {@inheritdoc} */
	public function getEditViewDisplayValue($value, $recordModel = false)
	{
		return '';
	}

	/** {@inheritdoc} */
	public function getApiDisplayValue($value, Vtiger_Record_Model $recordModel, array $params = [])
	{
		$refRecordModel = $this->getRecordModel($recordModel);
		if (null === $refRecordModel || empty($this->virtualFieldModel)) {
			return $value;
		}
		return $this->virtualFieldModel->getUITypeModel()->getApiDisplayValue(
			$refRecordModel->get($this->refFieldName),
			$refRecordModel,
			$params
		);
	}

	/** {@inheritdoc} */
	public function validate($value, $isUserFormat = false)
	{
		if (!empty($this->virtualFieldModel)) {
			$this->virtualFieldModel->getUITypeModel()->validate($value, $isUserFormat);
		}
	}

	/** {@inheritdoc} */
	public function getConditionBuilderField(string $operator): Vtiger_Field_Model
	{
		return $this->virtualFieldModel->getUITypeModel()->getConditionBuilderField($operator);
	}

	/** {@inheritdoc} */
	public function getDbConditionBuilderValue($value, string $operator)
	{
		if (empty($this->virtualFieldModel)) {
			return '';
		}
		return $this->virtualFieldModel->getUITypeModel()->getDbConditionBuilderValue($value, $operator);
	}

	/** {@inheritdoc} */
	public function getTemplateName()
	{
		if (empty($this->virtualFieldModel)) {
			return '';
		}
		return $this->virtualFieldModel->getUITypeModel()->getTemplateName();
	}

	/** {@inheritdoc} */
	public function getDetailViewTemplateName()
	{
		if (empty($this->virtualFieldModel)) {
			return '';
		}
		return $this->virtualFieldModel->getUITypeModel()->getDetailViewTemplateName();
	}

	/** {@inheritdoc} */
	public function getListSearchTemplateName()
	{
		if (empty($this->virtualFieldModel)) {
			return '';
		}
		return $this->virtualFieldModel->getUITypeModel()->getListSearchTemplateName();
	}

	/** {@inheritdoc} */
	public function getAllowedColumnTypes()
	{
		return $this->virtualFieldModel->getUITypeModel()->getAllowedColumnTypes();
	}

	/** {@inheritdoc} */
	public function isListviewSortable()
	{
		return $this->virtualFieldModel->getUITypeModel()->isListviewSortable();
	}

	/** {@inheritdoc} */
	public function getHeaderTypes()
	{
		return $this->virtualFieldModel->getUITypeModel()->getHeaderTypes();
	}

	/** {@inheritdoc} */
	public function getDefaultOperator()
	{
		return $this->virtualFieldModel->getUITypeModel()->getDefaultOperator();
	}

	/** {@inheritdoc} */
	public function getQueryOperators()
	{
		return $this->virtualFieldModel->getUITypeModel()->getQueryOperators();
	}

	/** {@inheritdoc} */
	public function getRecordOperators(): array
	{
		return $this->virtualFieldModel->getUITypeModel()->getRecordOperators();
	}

	/** {@inheritdoc} */
	public function getOperatorTemplateName(string $operator = '')
	{
		return $this->virtualFieldModel->getUITypeModel()->getOperatorTemplateName($operator);
	}

	/** {@inheritdoc} */
	public function getValueToExport($value, int $recordId)
	{
		return $this->virtualFieldModel->getUITypeModel()->getValueToExport($value, $recordId);
	}

	/** {@inheritdoc} */
	public function isResizableColumn(): bool
	{
		return $this->virtualFieldModel->getUITypeModel()->isResizableColumn();
	}

	/**
	 * Get virtual label.
	 *
	 * @return string
	 */
	public function getVirtualLabel(): string
	{
		$baseLabel = $this->getFieldModel()->getModule()->getFieldByName($this->baseField)->getFullLabelTranslation();
		$refModuleLabel = Language::translate($this->refModuleName, $this->refModuleName);
		$refFieldLabel = Vtiger_Module_Model::getInstance($this->refModuleName)
			->getFieldByName($this->refFieldName)->getFullLabelTranslation();
		return "$baseLabel ($refModuleLabel) > $refFieldLabel";
	}

	/**
	 * Function to get all the available picklist values for reference field.
	 *
	 * @return string[]
	 */
	public function getPicklistValues()
	{
		if (empty($this->virtualFieldModel)) {
			return [];
		}
		return $this->virtualFieldModel->getPicklistValues();
	}

	/**
	 * Get reference record model.
	 *
	 * @param Vtiger_Record_Model|bool|null $recordModel
	 *
	 * @return Vtiger_Record_Model|null
	 */
	protected function getRecordModel($recordModel): ?Vtiger_Record_Model
	{
		if (
			!$recordModel
			|| !($relId = $recordModel->getValueByField($this->baseField))
			|| !Record::isExists($relId, $this->refModuleName)
		) {
			return null;
		}
		if (!empty(self::$virtualRecordModel[$relId])) {
			return self::$virtualRecordModel[$relId];
		}
		if (!Record::isExists($relId, $this->refModuleName)) {
			return null;
		}
		return self::$virtualRecordModel[$relId] = Vtiger_Record_Model::getInstanceById($relId, $this->refModuleName);
	}

	/**
	 * There is no reason to compare virtual values, as the record does not store data.
	 * Preventing entries in the change history.
	 * {@inheritdoc}
	 */
	public function compare(mixed $value, mixed $prevValue): bool
	{
		return true;
	}
}
