<?php
/**
 * Mongo db connection file
 *
 * @package \App\Db
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author 	  Mateusz Slominski <m.slominski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Db;

use App\Config;
use App\Db\Mongo\Command;

/**
 * Mongo db connection class
 */
class Mongo extends \yii\mongodb\Connection
{
	/**
	 * @var \App\Db Table of connections with database
	 */
	private static $cache = [];

	/**
	 * @var array Configuration with database
	 */
	private static $config = [];

	/**
	 * @var string table/document name prefix
	 */
	public string $tablePrefix = '';

	/**
	 * Creates the Db connection instance.
	 *
	 * @param string $type Name of database connection
	 *
	 * @return self
	 */
	public static function getInstance(string $type = 'mongo')
	{
		if (isset(self::$cache[$type])) {
			return self::$cache[$type];
		}

		$db = new self(self::getConfig($type));
		$db->open();
		self::$cache[$type] = $db;
		return $db;
	}

	/**
	 * Load database connection configuration.
	 *
	 * @param string $type
	 *
	 * @return array with database configuration
	 */
	public static function getConfig(string $type) : array
	{
		if (!isset(self::$config[$type])) {
			self::$config[$type] = Config::db($type) ?? Config::db('mongo');
		}
		return self::$config[$type];
	}

	/**
	 * {@inheritDoc}
	 */
	public function getCollection($name, $refresh = false)
	{
		return parent::getCollection($this->resolveDocumentPrefix($name), $refresh);
	}

	/**
	 * {@inheritDoc}
	 */
	public function createCommand($document = [], $databaseName = null)
	{
		return new Command([
			'db' => $this,
			'databaseName' => $databaseName,
			'document' => $document,
			'globalExecOptions' => $this->globalExecOptions
		]);
	}

	/**
	 * Resolves document/table prefix
	 *
	 * @param $document
	 * @return array|mixed|string|string[]
	 */
	public function resolveDocumentPrefix($document)
	{
		if (is_array($document)) {
			$document[1] = str_replace('#__', $this->tablePrefix, $document[1]);
		} else {
			$document = str_replace('#__', $this->tablePrefix, $document);
		}

		return $document;
	}
}
