<?php

/**
 * Comarch account address synchronization file.
 *
 * The file is part of the paid functionality. Using the file is allowed only after purchasing a subscription.
 * File modification allowed only with the consent of the system producer.
 *
 * @package Integration
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

namespace App\Integrations\Comarch\Xl\Synchronizer;

/**
 * Comarch account address synchronization class.
 */
class AccountAddress extends \App\Integrations\Comarch\Synchronizer
{
	/**
	 * Get account address by account akronim from API.
	 *
	 * @param string $key Account akronim
	 *
	 * @return array
	 */
	public function getAddress(string $key): array
	{
		$cacheKey = $this::LOG_CATEGORY . '/AccountAddress';
		if (\App\Cache::staticHas($cacheKey, $key)) {
			return \App\Cache::staticGet($cacheKey, $key);
		}
		$address = [];
		try {
			if ($rows = $this->getFromApi('Address/GetByAkronim/' . $key)) {
				foreach ($rows as $row) {
					$mapModel = $this->getMapModel('AccountAddress');
					$mapModel->setDataApi($row);
					$address[$row[$mapModel::API_NAME_ID]] = $mapModel->getDataYf();
				}
			}
		} catch (\Throwable $ex) {
			$this->logError("getAddress {$this->name} by akronim", ['akronim' => $key, 'API' => $rows ?? []], $ex);
		}
		\App\Cache::staticSave($cacheKey, $key, $address);
		return $address;
	}

	/**
	 * Get account address by account akronim from API.
	 *
	 * @param array $address
	 *
	 * @return int|null
	 */
	public function createAddress(array $address): ?int
	{
		$mapModel = $this->getMapModel('AccountAddress', false);
		$mapModel->setDataApi([]);
		$mapModel->setDataYf($address);
		$mapModel->loadModeApi();
		$dataApi = $mapModel->getDataApi();
		if ($mapModel->skip) {
			if ($this->config->get('log_all')) {
				$this->controller->log(
					$this->name . ' ' . __FUNCTION__ . ' | skipped , inconsistent data',
					['YF' => $address, 'API' => $dataApi ?? []]
				);
			}
		} else {
			try {
				$response = $this->controller->getConnector()
					->request('POST', 'Address/Create', $dataApi);
				$response = \App\Json::decode($response);
			} catch (\Throwable $ex) {
				$this->logError("createAddress {$this->name} ", [
					'YF' => $address,
					'API' => $dataApi,
				], $ex);
				throw $ex;
			}
		}
		if ($this->config->get('log_all')) {
			$this->controller->log(
				$this->name . ' ' . __FUNCTION__ . ' | exported',
				[
					'YF' => $address,
					'API' => $dataApi,
					'response' => $response ?? null
				]
			);
		}
		return $response['id'] ?? $response['result']['knA_GidNumer'] ?? null;
	}
}
