<?php
/**
 * Mail account file.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Mail\Account\Entity;

use App\Encryption;
use App\Mail\Account\Entity\Enum\MailBoxType;
use App\Mail\Account\Entity\Enum\Status;
use App\Mail\Server;

/**
 * Mail account class.
 */
class AccountPrivateEntity implements EntityInterface
{
	private array $changes = [];
	private readonly ?int $id;
	private string $name;
	private string $login;
	private ?string $password;
	private ?string $refresh_token;
	private string $status;
	private ?string $logs;
	private ?string $expire_time;
	private int $mail_server_id;
	private ?string $scanner_actions;
	private ?string $scanner_fields;
	private ?string $domain_exceptions;
	private ?string $email_exceptions;
	private ?string $folders;
	private ?string $last_login;
	private int $owner_id;
	private readonly int $private;
	private readonly array $autologinUsers;

	public function setData(
		int $id,
		string $name,
		string $login,
		#[\SensitiveParameter] ?string $password,
		#[\SensitiveParameter] ?string $refresh_token,
		string $status,
		?string $logs,
		?string $expire_time,
		int $mail_server_id,
		?string $scanner_actions,
		?string $scanner_fields,
		?string $domain_exceptions,
		?string $email_exceptions,
		?string $folders,
		?string $last_login,
		int $owner_id,
		int $private,
		array $autologinUsers = []
	) {
		$this->id = $id;
		$this->name = $name;
		$this->login = $login;
		$this->password = (string) Encryption::getInstance(Encryption::TARGET_SETTINGS)->decrypt($password ?? '');
		$this->refresh_token = (string) Encryption::getInstance(Encryption::TARGET_SETTINGS)->decrypt($refresh_token ?? '');
		$this->status = $status;
		$this->logs = $logs;
		$this->expire_time = $expire_time;
		$this->mail_server_id = $mail_server_id;
		$this->scanner_actions = $scanner_actions;
		$this->scanner_fields = $scanner_fields;
		$this->domain_exceptions = $domain_exceptions;
		$this->email_exceptions = $email_exceptions;
		$this->folders = $folders;
		$this->last_login = $last_login;
		$this->owner_id = $owner_id;
		$this->private = $private;
		$this->autologinUsers = $autologinUsers;

		return $this;
	}

	public function set(string $key, mixed $value)
	{
		if (property_exists($this, $key)) {
			$this->setChanges($key, $value);
			$this->{$key} = $value;
		}

		return $this;
	}

	public function get(string $key): mixed
	{
		return $this->{$key} ?? '';
	}

	public function has(string $key)
	{
		return isset($this->{$key});
	}

	public function getId(): ?int
	{
		return $this->id ?? null;
	}

	public function setId(int $id)
	{
		$this->id ??= $id;

		return $this;
	}

	public function getName(): string
	{
		return $this->name;
	}

	public function getLogin(): string
	{
		return $this->login;
	}

	public function setRefreshToken(#[\SensitiveParameter] string $token)
	{
		$this->setChanges('refresh_token', $token);
		$this->refresh_token = $token;

		return $this;
	}

	public function getRefreshToken(): string
	{
		return $this->refresh_token ?? '';
	}

	public function getServer(): Server
	{
		return Server::getInstanceById($this->mail_server_id);
	}

	public function setPassword(#[\SensitiveParameter] string $token)
	{
		$this->setChanges('password', $token);
		$this->password = $token;

		return $this;
	}

	public function getPassword(): string
	{
		return $this->password ?? '';
	}

	/**
	 * Set expire time.
	 *
	 * @param string $expireTime DateTime as string
	 */
	public function setExpireTime(string $expireTime): self
	{
		$this->setChanges('expire_time', $expireTime);
		$this->expire_time = $expireTime;

		return $this;
	}

	public function setLastLogin(string $loginTime)
	{
		$this->setChanges('last_login', $loginTime);
		$this->last_login = $loginTime;

		return $this;
	}

	/**
	 * Return the last login time.
	 *
	 * @return string|null DateTime as string eg. (Y-m-d H:i:s)
	 */
	public function getLastLogin(): ?string
	{
		return $this->last_login ?? null;
	}

	public function setLogs(string $message)
	{
		$messages = \App\Purifier::decodeHtml(\App\Purifier::encodeHtml($message));
		$messages = \App\TextUtils::textTruncate($messages, 65535, true, true);
		$this->setChanges('logs', $messages);
		$this->logs = $messages;

		return $this;
	}

	public function getLogs(): string
	{
		return $this->logs ?? '';
	}

	public function setStatus(Status $status)
	{
		$this->setChanges('status', $status->value);
		$this->status = $status->value;

		return $this;
	}

	public function getStatus(): ?Status
	{
		return Status::tryFrom($this->status);
	}

	public function getOwnerId(): int
	{
		return $this->owner_id;
	}

	public function isPrivate(): bool
	{
		return MailBoxType::PRIVATE->value === $this->private;
	}

	public function setChanges(string $key, mixed $value)
	{
		if (!isset($this->{$key}) || $this->{$key} !== $value) {
			$this->changes[$key] = $this->{$key} ?? null;
		}

		return $this;
	}

	public function getValuesForSave(): array
	{
		$forSave = [];
		foreach ($this->getChanges() as $fieldName => $prevValue) {
			switch ($fieldName) {
				case 'refresh_token':
				case 'password':
					$forSave[$fieldName] = \App\Encryption::getInstance()->encrypt($this->{$fieldName});
					break;
				default:
					$forSave[$fieldName] = $this->{$fieldName};
					break;
			}
		}

		return $forSave;
	}

	public function getChanges(): array
	{
		return $this->changes;
	}

	public function getFolders(): array
	{
		return isset($this->folders) ? \App\Json::decode($this->folders) : [];
	}

	public function getAutologinUsers(): array
	{
		return $this->autologinUsers;
	}

	public function setScannerActions(string $actions)
	{
		$this->scanner_actions = $actions;

		return $this;
	}

	public function setScannerFields(string $fields)
	{
		$this->scanner_fields = $fields;

		return $this;
	}
}
