<?php
/**
 * Base driver class.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Klaudia Łozowska <k.lozowska@yetiforce.com>
 */

declare(strict_types=1);

namespace App\Map;

use App\Encryption;

/**
 * Base driver class.
 */
abstract class BaseDriver
{
	/**
	 * Label.
	 *
	 * @var string
	 */
	protected string $label;

	/**
	 * Configuration form fields.
	 *
	 * @var array
	 */
	protected array $formFields;

	/**
	 * Construct.
	 *
	 * @param array $config
	 */
	public function __construct(private array $config)
	{
		if ($params = ($config['params'] ?? null)) {
			$this->config['params'] = json_decode($params, true);
		}

		if ($apiKey = $this->config['api_key'] ?? null) {
			$this->config['api_key'] = Encryption::getInstance()->decrypt($apiKey);
		}
	}

	/**
	 * Get label.
	 *
	 * @return string
	 */
	public function getLabel(): string
	{
		return $this->label;
	}

	/**
	 * Get configuration form fields.
	 *
	 * @return array
	 */
	public function getFormFields(): array
	{
		$fields = [];
		foreach ($this->formFields as $fieldName => $configData) {
			if (isset($this->getConfig()[$fieldName])) {
				$configData['fieldvalue'] = $this->getConfig()[$fieldName];
			}
			if (isset($this->getConfig()['params'][$fieldName])) {
				$configData['fieldvalue'] = $this->getConfig()['params'][$fieldName];
			}
			$fields[$fieldName] = \Vtiger_Field_Model::init('Settings:Map', $configData, $fieldName);
		}

		return $fields;
	}

	/**
	 * Get configuration.
	 *
	 * @return array
	 */
	public function getConfig(): array
	{
		return $this->config;
	}
}
