<?php

/**
 * Generate report xls file.
 *
 * @package   Actions
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Lukasz Krawczyk <l.krawczyk@yetiforce.com>
 * @author    Adrian Kon <a.kon@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

use App\Controller\Action;
use App\DocumentBuilder\FileGenerator;
use App\DocumentBuilder\Generator\XlsxGenerator;
use App\DocumentBuilder\Integrations\DocumentModuleIntegration;
use App\Exceptions\NoPermittedToRecord;
use App\Language;
use App\Log;
use App\Privilege;
use App\Report\Provider\FieldsProvider;
use App\Report\Provider\ReportProvider;
use App\Report\Report;
use App\Report\Repository\TemplateRepository;
use App\Request;
use App\User;

/** Generate report class */
final class ReportTemplate_GenerateReport_Action extends Action
{
	/** {@inheritdoc} */
	public function checkPermission(Request $request)
	{
		$recordId = $request->isEmpty('record', true) ? 0 : $request->getInteger('record');
		if (!$recordId
			|| !Privilege::isPermitted($request->getModule(), 'DetailView', $recordId)
			|| !Privilege::isPermitted('Documents', 'CreateView')
		) {
			throw new NoPermittedToRecord('ERR_NO_PERMISSIONS_FOR_THE_RECORD', 406);
		}
	}

	/** {@inheritdoc} */
	public function process(Request $request): void
	{
		$response = new Vtiger_Response();
		$moduleName = $request->getModule();

		try {
			$report = Report::get(TemplateRepository::class)->find($request->getInteger('record'));
			$fileGenerator = (new FileGenerator())->setFileGenerator(new XlsxGenerator());
			$fieldsProvider = Report::get(FieldsProvider::class);
			$dataProvider = Report::get(ReportProvider::class);

			$data = $dataProvider->provideToXls($report->getId(), User::getCurrentUserId());
			$headers = $fieldsProvider->provide($report->getId(), \count($data) > 0 ? $data[0] : []);

			$fileGenerator->addData($data)->generateFile(
				$report,
				$headers,
				$fieldsProvider->wrappedFields($report->getId()),
			);

			$filePath = $fileGenerator->getFilePath($fileGenerator->getReportFilename($report));

			$document = (new DocumentModuleIntegration())->createDocumentEntry($report, $filePath);

			$result = [
				'success' => true,
				'message' => Language::translateArgs('LBL_REPORT_GENERATED_SUCCESS', $moduleName, $report->getName()),
				'file' => $document->getDownloadFileURL(),
			];
		} catch (Exception $exception) {
			Log::error($exception->__toString());
			$result = [
				'success' => false,
				'message' => Language::translateArgs('LBL_REPORT_GENERATED_FAIL', $moduleName, $report->getName()),
			];
		}

		$response->setResult($result);
		$response->emit();
	}
}
