<?php

/**
 * Mail folders modal view file.
 *
 * @package View
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\Controller\Modal;
use App\Exceptions\NoPermittedToRecord;
use App\Json;
use App\Log;
use App\Purifier;
use App\Request;
use App\User;

/**
 * Mail folders modal view class.
 */
class Users_MailFoldersModal_View extends Modal
{
	/** {@inheritdoc} */
	public $pageTitle = 'FL_MAILBOX_FOLDERS';
	/** {@inheritdoc} */
	public $successBtn = 'LBL_SELECT_OPTION';

	/** {@inheritdoc} */
	public function checkPermission(Request $request)
	{
		$recordId = $request->getInteger('template');
		$baseId = User::getCurrentUserRealId();
		$currentUserId = User::getCurrentUserId();
		if ($currentUserId !== $recordId || ($baseId !== $currentUserId && !User::getUserModel($baseId)->isAdmin())) {
			throw new NoPermittedToRecord('ERR_NO_PERMISSIONS_FOR_THE_RECORD', 406);
		}
	}

	/**
	 * Tree in popup.
	 *
	 * @param Request $request
	 */
	public function process(Request $request)
	{
		$moduleName = $request->getModule(false);
		$recordId = $request->getInteger('template');
		$mailboxModel = new Users_Mailbox_Model($recordId);

		$missingFolders = $selectedFolders = $folders = $mailScannerFolders = [];
		$foldersFlat = [];
		$error = '';

		try {
			if (!$mailboxModel->isExists()
				|| ($mailboxModel->getAccount()->getServer()->isOAuth() && !$mailboxModel->getAccount()->getRefreshToken())
			) {
				$error = 'ERR_IMAP_CONNECTION';
			} else {
				$mailAccount = $mailboxModel->getAccount();
				$imap = $mailAccount->openImap();
				$folders = $imap->getFolders();
				$mailScannerFolders = $mailAccount->getFolders();
			}
		} catch (Throwable $th) {
			Log::error($th->__toString());
			$error = 'ERR_IMAP_CONNECTION';
		}

		array_walk_recursive($folders, static function ($value, $k) use (&$foldersFlat) {
			if ('fullName' === $k) {
				$foldersFlat[] = $value;
			}
		});

		foreach ($mailScannerFolders as $folder) {
			if (!\in_array($folder, $foldersFlat)) {
				$missingFolders[] = $folder;
			} else {
				$selectedFolders[] = $folder;
			}
		}

		if ($error) {
			$this->successBtn = null;
		}

		$viewer = $this->getViewer($request);
		$viewer->assign('TREE', Json::encode($this->getTreeData($folders, $selectedFolders)));
		$viewer->assign('IS_MULTIPLE', true);
		$viewer->assign('MISSING_FOLDERS', $missingFolders);
		$viewer->assign('ERROR', $error);
		$viewer->view('Modals/MailFoldersModal.tpl', $moduleName);
	}

	/**
	 * Get tree item.
	 *
	 * @param array       $folder
	 * @param array       $selectedFolders
	 * @param array       $tree
	 * @param string|null $parent
	 *
	 * @return array
	 */
	public function getTreeItem(array $folder, array $selectedFolders, array &$tree, ?string $parent = null): array
	{
		$selected = \in_array($folder['fullName'], (array) $selectedFolders);
		$treeRecord = [
			'id' => $folder['fullName'],
			'type' => 'record',
			'parent' => $parent ?? '#',
			'text' => Purifier::encodeHtml($folder['name']),
			'state' => [
				'selected' => $selected,
				'opened' => true,
				'disabled' => !$selected && ($folder['noSelect'] ?? false)
			]
		];
		$treeRecord['db_id'] = $folder['fullName'];

		$tree[] = $treeRecord;
		if (!empty($folder['children'])) {
			foreach ($folder['children'] as $subfolder) {
				$this->getTreeItem($subfolder, $selectedFolders, $tree, $folder['fullName']);
			}
		}

		return $treeRecord;
	}

	/** {@inheritdoc} */
	public function getModalScripts(Request $request)
	{
		$jsFileNames = ['~libraries/jstree/dist/jstree.js'];
		$jsFileNames[] = '~layouts/resources/libraries/jstree.category.js';
		$jsFileNames[] = '~layouts/resources/libraries/jstree.checkbox.js';
		$jsFileNames[] = 'modules.Vtiger.resources.TreeModal';
		$jsFileNames[] = 'modules.Vtiger.resources.MailFoldersModal';

		return array_merge($this->checkAndConvertJsScripts($jsFileNames), parent::getModalScripts($request));
	}

	/** {@inheritdoc} */
	public function getModalCss(Request $request)
	{
		return array_merge(parent::getModalCss($request), $this->checkAndConvertCssStyles([
			'~libraries/jstree-bootstrap-theme/dist/themes/proton/style.css',
		]));
	}

	/** {@inheritdoc} */
	protected function preProcessTplName(Request $request)
	{
		return 'Modals/TreeHeader.tpl';
	}

	/**
	 * Get tree data.
	 *
	 * @param array $folders
	 * @param array $selectedFolders
	 *
	 * @return array
	 */
	private function getTreeData(array $folders, array $selectedFolders): array
	{
		$tree = [];
		foreach ($folders as $folder) {
			$this->getTreeItem($folder, $selectedFolders, $tree, null);
		}

		return $tree;
	}
}
