<?php

use App\Fields\Double;
use App\Json;
use App\Purifier;
use App\Validator;

/**
 * Tax percent field.
 *
 * @package   InventoryField
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

/**
 * Inventory TaxPercent Field Class.
 */
class Vtiger_TaxPercent_InventoryField extends Vtiger_Tax_InventoryField
{
	/** @var array List of shared fields */
	public $shared = ['taxparam' => 'tax'];
	protected $type = 'TaxPercent';
	protected $defaultLabel = 'LBL_TAX_IN_PERCENT';
	protected $defaultValue = 0;
	protected $summationValue = false;
	protected $columnName = 'tax_percent';
	protected $dbType = 'decimal(12,8) DEFAULT 0';
	protected $maximumLength = '9999';
	protected $purifyType = Purifier::NUMBER;

	/** {@inheritdoc} */
	public function getDisplayValue($value, array $rowData = [], bool $rawText = false)
	{
		$taxPercentName = $this->getTaxPercentName($rowData, $rawText);
		if ($taxPercentName) {
			return $taxPercentName;
		}
		$value = Double::formatToDisplay($value);
		return $rawText ? $value : $value . '%';
	}

	public function getDefaultValue(string $columnName = ''): mixed
	{
		$value = parent::getDefaultValue($columnName);
		return $value ?: (Vtiger_Inventory_Model::getDefaultGlobalTax()['value'] ?? '');
	}

	/**
	 * Get selected tax name.
	 *
	 * @param array $rowData
	 * @param bool  $rawText
	 *
	 * @return string
	 */
	public function getTaxPercentName(array $rowData = [], bool $rawText = false): string
	{
		$result = '';
		if (!Json::isEmpty($rowData['taxparam'] ?? '')) {
			$taxParam = Json::decode($rowData['taxparam']);
			$types = (array) $taxParam['aggregationType'];
			$taxIds = [];
			foreach ($types as $type) {
				if (!empty($taxParam["{$type}Id"])) {
					$taxIds[] = $taxParam["{$type}Id"];
				}
			}

			if (\count($taxIds) > 0) {
				$globalTaxes = Vtiger_Inventory_Model::getGlobalTaxes();
				$taxNames = array_filter(array_map(
					static fn ($id) => isset($globalTaxes[$id]) && 1 === $globalTaxes[$id]['show_name']
						? $globalTaxes[$id]['name']
						: '',
					$taxIds
				));

				if ($taxNames) {
					$result = $rawText
						? implode(' ', $taxNames)
						: \App\Purifier::encodeHtml(implode(' ', $taxNames));
				}
			}
		}
		return $result;
	}

	/** {@inheritdoc} */
	public function getValueForSave(array $item, bool $userFormat = false, ?string $column = null)
	{
		if ($column === $this->getColumnName() || null === $column) {
			$value = 0.0;
			if (!Json::isEmpty($item['taxparam'] ?? '')) {
				$taxParam = Json::decode($item['taxparam']);
				$types = (array) $taxParam['aggregationType'];
				foreach ($types as $type) {
					$value += $taxParam["{$type}Tax"];
				}
			}
		} else {
			$value = $item[$column] ?? $this->getDefaultValue($column);
			$value = $userFormat ? $this->getDBValue($value) : $value;
		}
		return $value;
	}

	/** {@inheritdoc} */
	public function compare($value, $prevValue, string $column): bool
	{
		return Validator::floatIsEqual((float) $value, (float) $prevValue, 8);
	}

	/** {@inheritdoc} */
	public function getConfigFieldsData(): array
	{
		$data = parent::getConfigFieldsData();
		$data['summary_enabled'] = [
			'name' => 'summary_enabled',
			'label' => 'LBL_INV_SUMMARY_ENABLED',
			'uitype' => 56,
			'maximumlength' => '1',
			'typeofdata' => 'C~O',
			'purifyType' => Purifier::INTEGER,
			'defaultvalue' => 1
		];

		return $data;
	}
}
